<?php
/**
 * سیستم جامع مدیریت خطا برای پنل مرزبان
 * این فایل تمام خطاهای مربوط به پنل را مدیریت می‌کند
 */

/**
 * بررسی اعتبار token و refresh در صورت نیاز
 * @param string $code_panel کد پنل
 * @param string $location نام پنل
 * @return array ['success' => bool, 'token' => string|null, 'error' => string|null]
 */
function ensureValidToken($code_panel, $location = '') {
    global $pdo;
    
    // تلاش اول با verify
    $tokenResult = token_panel($code_panel, true);
    
    // اگر خطا داشت یا token نداشت، force refresh
    if (!empty($tokenResult['error']) || !isset($tokenResult['access_token'])) {
        logError("ensureValidToken", "Token نامعتبر، در حال refresh", [
            'code_panel' => $code_panel,
            'location' => $location,
            'error' => $tokenResult['error'] ?? 'no token'
        ]);
        
        // Force refresh: حذف cache و دریافت token جدید
        $panel = select("marzban_panel", "*", "code_panel", $code_panel, "select");
        if ($panel && isset($panel['name_panel'])) {
            update("marzban_panel", "datelogin", null, "name_panel", $panel['name_panel']);
        }
        
        // دریافت token جدید بدون verify
        $tokenResult = token_panel($code_panel, false);
    }
    
    if (!empty($tokenResult['error']) || !isset($tokenResult['access_token'])) {
        return [
            'success' => false,
            'token' => null,
            'error' => $tokenResult['error'] ?? 'خطا در دریافت token'
        ];
    }
    
    return [
        'success' => true,
        'token' => $tokenResult['access_token'],
        'error' => null
    ];
}

/**
 * بررسی و مدیریت خطاهای getuser با retry و token refresh
 * @param array $response پاسخ getuser
 * @param string $username نام کاربری
 * @param string $panel_name نام پنل
 * @param string $code_panel کد پنل
 * @param int $retry_count تعداد تلاش مجدد
 * @return array ['success' => bool, 'message' => string, 'data' => array|null]
 */
function handleGetUserError($response, $username = '', $panel_name = '', $code_panel = '', $retry_count = 0) {
    // بررسی اولیه: آیا پاسخ یک آرایه است؟
    if (!is_array($response)) {
        logError("getuser", "پاسخ نامعتبر از پنل", [
            'username' => $username,
            'panel' => $panel_name,
            'response_type' => gettype($response)
        ]);
        return [
            'success' => false,
            'message' => "❌ خطا در ارتباط با پنل. لطفاً اتصال پنل را بررسی کنید.",
            'data' => null
        ];
    }
    
    // بررسی خطای curl
    if (!empty($response['error'])) {
        logError("getuser", "خطای اتصال", [
            'username' => $username,
            'panel' => $panel_name,
            'error' => $response['error']
        ]);
        
        $errorMsg = strtolower($response['error']);
        if (strpos($errorMsg, 'timeout') !== false || strpos($errorMsg, 'timed out') !== false) {
            return [
                'success' => false,
                'message' => "⏱️ زمان اتصال به پنل به پایان رسید. لطفاً دوباره تلاش کنید.",
                'data' => null
            ];
        } elseif (strpos($errorMsg, 'connection') !== false || strpos($errorMsg, 'resolve') !== false) {
            return [
                'success' => false,
                'message' => "🔌 امکان اتصال به پنل وجود ندارد. لطفاً آدرس پنل را بررسی کنید.",
                'data' => null
            ];
        } else {
            return [
                'success' => false,
                'message' => "❌ خطا در ارتباط با پنل: " . $response['error'],
                'data' => null
            ];
        }
    }
    
    // بررسی کد وضعیت HTTP
    if (!empty($response['status'])) {
        $status = (int)$response['status'];
        
        if ($status == 404) {
            logError("getuser", "کاربر پیدا نشد (404)", [
                'username' => $username,
                'panel' => $panel_name
            ]);
            return [
                'success' => false,
                'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 راهنمایی:\n• مطمئن شوید نام کاربری را به درستی وارد کرده‌اید\n• نام کاربری باید دقیقاً همان چیزی باشد که در پنل وجود دارد\n• حروف بزرگ و کوچک مهم هستند",
                'data' => null
            ];
        } elseif ($status == 401 || $status == 403) {
            logError("getuser", "خطای احراز هویت", [
                'username' => $username,
                'panel' => $panel_name,
                'status' => $status,
                'retry_count' => $retry_count
            ]);
            
            // اگر خطای 401/403 بود و retry نکرده‌ایم، token را refresh کنیم و دوباره تلاش کنیم
            if ($retry_count < 1 && !empty($code_panel)) {
                $tokenCheck = ensureValidToken($code_panel, $panel_name);
                if ($tokenCheck['success']) {
                    // دوباره getuser را صدا بزنیم
                    require_once __DIR__ . '/Marzban.php';
                    $newResponse = getuser($username, $panel_name);
                    return handleGetUserError($newResponse, $username, $panel_name, $code_panel, $retry_count + 1);
                }
            }
            
            return [
                'success' => false,
                'message' => "🔐 خطا در احراز هویت پنل.\n\n💡 راهنمایی:\n• لطفاً نام کاربری و رمز عبور پنل را بررسی کنید\n• ممکن است token منقضی شده باشد - دوباره تلاش کنید\n• اگر مشکل ادامه داشت، پنل را restart کنید",
                'data' => null
            ];
        } elseif ($status == 500) {
            logError("getuser", "خطای سرور پنل (500)", [
                'username' => $username,
                'panel' => $panel_name
            ]);
            return [
                'success' => false,
                'message' => "⚠️ خطای داخلی سرور پنل. لطفاً چند لحظه صبر کرده و دوباره تلاش کنید.",
                'data' => null
            ];
        } elseif ($status != 200) {
            logError("getuser", "خطای HTTP", [
                'username' => $username,
                'panel' => $panel_name,
                'status' => $status
            ]);
            return [
                'success' => false,
                'message' => "❌ خطا در ارتباط با پنل (کد خطا: {$status})",
                'data' => null
            ];
        }
    }
    
    // بررسی body
    if (empty($response['body'])) {
        logError("getuser", "پاسخ خالی از پنل", [
            'username' => $username,
            'panel' => $panel_name
        ]);
        return [
            'success' => false,
            'message' => "❌ پاسخ خالی از پنل دریافت شد. لطفاً اتصال پنل را بررسی کنید.",
            'data' => null
        ];
    }
    
    // تبدیل JSON به آرایه
    $data = json_decode($response['body'], true);
    if ($data === null || !is_array($data)) {
        logError("getuser", "خطا در پردازش JSON", [
            'username' => $username,
            'panel' => $panel_name,
            'body_preview' => substr($response['body'], 0, 200)
        ]);
        return [
            'success' => false,
            'message' => "❌ خطا در پردازش پاسخ پنل. لطفاً دوباره تلاش کنید.",
            'data' => null
        ];
    }
    
    // بررسی پیام خطا در JSON
    if (isset($data['msg']) && strtolower($data['msg']) == "user not found") {
        logError("getuser", "کاربر پیدا نشد (از JSON)", [
            'username' => $username,
            'panel' => $panel_name
        ]);
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 راهنمایی:\n• مطمئن شوید نام کاربری را به درستی وارد کرده‌اید\n• نام کاربری باید دقیقاً همان چیزی باشد که در پنل وجود دارد\n• حروف بزرگ و کوچک مهم هستند",
            'data' => null
        ];
    }
    
    if (isset($data['detail']) && !empty($data['detail'])) {
        $detail = strtolower($data['detail']);
        if (strpos($detail, 'not found') !== false || strpos($detail, 'پیدا نشد') !== false) {
            logError("getuser", "کاربر پیدا نشد (از detail)", [
                'username' => $username,
                'panel' => $panel_name,
                'detail' => $data['detail']
            ]);
            return [
                'success' => false,
                'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 راهنمایی:\n• مطمئن شوید نام کاربری را به درستی وارد کرده‌اید\n• نام کاربری باید دقیقاً همان چیزی باشد که در پنل وجود دارد",
                'data' => null
            ];
        }
        logError("getuser", "خطا از detail", [
            'username' => $username,
            'panel' => $panel_name,
            'detail' => $data['detail']
        ]);
        return [
            'success' => false,
            'message' => "❌ خطا: " . $data['detail'],
            'data' => null
        ];
    }
    
    // بررسی وجود username برای اطمینان از اینکه کاربر پیدا شده
    if (!isset($data['username']) || empty($data['username'])) {
        logError("getuser", "username در پاسخ وجود ندارد", [
            'username' => $username,
            'panel' => $panel_name,
            'available_fields' => array_keys($data)
        ]);
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 این خطا ممکن است به این دلایل باشد:\n• نام کاربری در پنل وجود ندارد\n• پاسخ پنل نامعتبر است",
            'data' => null
        ];
    }
    
    // بررسی وجود فیلدهای ضروری برای تنظیم اینباند
    // اگر proxies یا proxy_settings وجود نداشت، اما username وجود داشت، یعنی کاربر هست ولی اینباند تنظیم نشده
    if (!isset($data['proxies']) && !isset($data['proxy_settings'])) {
        logError("getuser", "فیلدهای proxies/proxy_settings وجود ندارد", [
            'username' => $username,
            'panel' => $panel_name,
            'user_found' => isset($data['username']),
            'available_fields' => array_keys($data)
        ]);
        
        // اگر username وجود داشت، یعنی کاربر پیدا شده ولی اینباند ندارد
        if (isset($data['username']) && $data['username'] == $username) {
            return [
                'success' => false,
                'message' => "⚠️ کاربر در پنل پیدا شد اما اینباند تنظیم نشده است!\n\n💡 راهنمایی:\n• ابتدا در پنل مرزبان برای این کاربر یک اینباند فعال کنید\n• سپس دوباره نام کاربری را ارسال کنید\n• مطمئن شوید که پروتکل و اینباند برای این کاربر فعال است",
                'data' => null
            ];
        }
        
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 این خطا ممکن است به این دلایل باشد:\n• نام کاربری در پنل وجود ندارد\n• کانفیگ اینباند برای این کاربر تنظیم نشده\n• پنل به درستی پاسخ نمی‌دهد",
            'data' => null
        ];
    }
    
    // همه چیز درست است
    return [
        'success' => true,
        'message' => '',
        'data' => $data
    ];
}

/**
 * بررسی و مدیریت خطاهای DataUser
 * @param array $response پاسخ DataUser
 * @param string $username نام کاربری
 * @param string $panel_name نام پنل
 * @return array ['success' => bool, 'message' => string, 'data' => array|null]
 */
function handleDataUserErrorForAdmin($response, $username = '', $panel_name = '') {
    // بررسی اولیه
    if (!is_array($response)) {
        logError("DataUser", "پاسخ نامعتبر", [
            'username' => $username,
            'panel' => $panel_name
        ]);
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!",
            'data' => null
        ];
    }
    
    // بررسی خطا
    if (isset($response['status']) && $response['status'] == 'Unsuccessful') {
        $errorMsg = isset($response['msg']) ? strtolower(trim($response['msg'])) : '';
        
        $isUserNotFound = (
            $errorMsg == "user not found" ||
            strpos($errorMsg, "user not found") !== false ||
            strpos($errorMsg, "کاربر پیدا نشد") !== false ||
            strpos($errorMsg, "سرویس پیدا نشد") !== false ||
            strpos($errorMsg, "not found") !== false ||
            empty($errorMsg)
        );
        
        if ($isUserNotFound) {
            logError("DataUser", "کاربر پیدا نشد", [
                'username' => $username,
                'panel' => $panel_name,
                'msg' => $response['msg'] ?? ''
            ]);
            return [
                'success' => false,
                'message' => "❌ سرویس موردنظر در سرور پیدا نشد!\n\n💡 راهنمایی:\n• مطمئن شوید نام کاربری را به درستی وارد کرده‌اید\n• نام کاربری باید دقیقاً همان چیزی باشد که در پنل وجود دارد",
                'data' => null
            ];
        }
        
        logError("DataUser", "خطای دیگر", [
            'username' => $username,
            'panel' => $panel_name,
            'msg' => $response['msg'] ?? ''
        ]);
        return [
            'success' => false,
            'message' => isset($response['msg']) && !empty($response['msg']) ? "❌ " . $response['msg'] : "❌ سرویس موردنظر در سرور پیدا نشد!",
            'data' => null
        ];
    }
    
    // بررسی فیلدهای ضروری
    if (!array_key_exists('data_limit', $response) || !array_key_exists('used_traffic', $response)) {
        logError("DataUser", "فیلدهای ضروری وجود ندارد", [
            'username' => $username,
            'panel' => $panel_name
        ]);
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!",
            'data' => null
        ];
    }
    
    return [
        'success' => true,
        'message' => '',
        'data' => $response
    ];
}

/**
 * ثبت خطا در لاگ
 * @param string $function نام تابع
 * @param string $message پیام خطا
 * @param array $context اطلاعات اضافی
 */
function logError($function, $message, $context = []) {
    $logFile = __DIR__ . '/storage/cache/error_log_' . date('Y-m-d') . '.txt';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    $logEntry = [
        'time' => date('Y-m-d H:i:s'),
        'function' => $function,
        'message' => $message,
        'context' => $context,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ];
    
    $logLine = json_encode($logEntry, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) . "\n" . str_repeat('-', 80) . "\n";
    
    @file_put_contents($logFile, $logLine, FILE_APPEND);
}

/**
 * بررسی و مدیریت خطاهای getuserm (مرزنشین)
 * @param array $response پاسخ getuserm
 * @param string $username نام کاربری
 * @param string $panel_name نام پنل
 * @return array ['success' => bool, 'message' => string, 'data' => array|null]
 */
function handleGetUserMError($response, $username = '', $panel_name = '') {
    if (!is_array($response)) {
        return [
            'success' => false,
            'message' => "❌ خطا در ارتباط با پنل مرزنشین",
            'data' => null
        ];
    }
    
    if (!empty($response['error'])) {
        return [
            'success' => false,
            'message' => "❌ خطا: " . $response['error'],
            'data' => null
        ];
    }
    
    if (!empty($response['status']) && $response['status'] != 200) {
        return [
            'success' => false,
            'message' => "❌ خطا در ارتباط با پنل (کد: {$response['status']})",
            'data' => null
        ];
    }
    
    if (empty($response['body'])) {
        return [
            'success' => false,
            'message' => "❌ پاسخ خالی از پنل دریافت شد",
            'data' => null
        ];
    }
    
    $data = json_decode($response['body'], true);
    if ($data === null || !is_array($data)) {
        return [
            'success' => false,
            'message' => "❌ خطا در پردازش پاسخ پنل",
            'data' => null
        ];
    }
    
    if (isset($data['detail']) && strtolower($data['detail']) == "user not found") {
        return [
            'success' => false,
            'message' => "❌ سرویس موردنظر در سرور پیدا نشد!",
            'data' => null
        ];
    }
    
    return [
        'success' => true,
        'message' => '',
        'data' => $data
    ];
}

